using System.Collections.Generic;
using System.Net.Http;
using System.Threading.Tasks;
using Microsoft.Azure.WebJobs;
using Microsoft.Azure.WebJobs.Extensions.DurableTask;
using Microsoft.Azure.WebJobs.Extensions.Http;
using Microsoft.Azure.WebJobs.Host;
using Microsoft.Extensions.Logging;
using Microsoft.AspNetCore.Mvc;
using Newtonsoft.Json;
using Microsoft.WindowsAzure.Storage.Blob;
using Amazon.S3;
using Amazon.S3.Transfer;
using Amazon.S3.Util;
using System;
using Microsoft.AspNetCore.Http;
using Microsoft.WindowsAzure.Storage.Auth;
using Amazon.S3.Model;
//curl -X POST http://localhost:7071/api/TransferAWS_HttpStart -H 'Content-Type: application/json' -d "{\"ADFsettings\": [{\"SettingName\":\"blobUri\",\"SettingVal\":\"https://storageaccountxx.blob.core.windows.net/container/filenamex.csv\"},{\"SettingName\": \"bucketName\",\"SettingVal\":\"hostname-us-east-1/incoming\"}]}"

namespace FunctionAWS
{
     public static class TransferAWS
    {
        public static string blobUri;
        public static string bucketName;

        [FunctionName("TransferAWS")]
        public static async Task<List<string>> RunOrchestrator(
            [OrchestrationTrigger] IDurableOrchestrationContext context)
        {
            var outputs = new List<string>();
            ADFRequest data = context.GetInput<ADFRequest>();
            // run Durable Activity Function.
            outputs.Add(await context.CallActivityAsync<string>("TransferAWS_Proc", data));
            return outputs;
        }

        [FunctionName("TransferAWS_Proc")]
        public static string awsProc([ActivityTrigger] ADFRequest ADFReq, ILogger log)

        {       
            foreach (var adfset in ADFReq.ADFsettings)
            {
                log.LogInformation("ActivityFunction reading:" + adfset.SettingName);
                if (adfset.SettingName== "blobUri")
                {
                    blobUri = adfset.SettingVal.Trim();
                }
                if (adfset.SettingName == "bucketName")
                {
                    bucketName = adfset.SettingVal.Trim();
                }
            }

            log.LogInformation("AWS transfer begins:" + blobUri);

            string storageacct = Environment.GetEnvironmentVariable("storageaccount");
            string storagekey = Environment.GetEnvironmentVariable("accountkey");
            // Validate query string
            if (String.IsNullOrEmpty(blobUri) || String.IsNullOrEmpty(bucketName))
            {
                Result outcome = new Result("Invalid Parameters Passed to Function", false, "blobUri or bucketName is null or empty");
                return outcome.ConvertResultToJson();
            }
            //StorageCredentials
            StorageCredentials stc = new StorageCredentials(storageacct, storagekey);
            // cast the blob to its type
            Uri blobAbsoluteUri = new Uri(blobUri);
            CloudAppendBlob blob = new CloudAppendBlob(blobAbsoluteUri, stc);
            // CloudAppendBlob version ONLY, not for block blob types
          
            // Do the Copy
            Task<bool> resultBool =  CopyBlobAsync(blob, bucketName, log);
            if (resultBool.Result)
            {
                Result outcome = new Result("Copy Completed", true, "Blob: " + blobUri + " Copied to Bucket: " + bucketName);
                return outcome.ConvertResultToJson();
            }
            else
            {
                Result outcome = new Result("ERROR", false, "Copy Errored!");
                return outcome.ConvertResultToJson();
            }

        }

        private static async Task<bool> CopyBlobAsync(CloudAppendBlob blob, string existingBucket, ILogger log)
        {
            string accessKey = Environment.GetEnvironmentVariable("awsaccessKey");
            string secretKey = Environment.GetEnvironmentVariable("awssecretKey");

             var keyName = blob.Name;  
                      
            // Make the client 
            AmazonS3Client myClient = new AmazonS3Client(accessKey, secretKey, Amazon.RegionEndpoint.USEast1);

            // Check the Target Bucket Exists; 
          Task<bool> bucketExists =  AmazonS3Util.DoesS3BucketExistAsync(myClient, existingBucket);

           // bucketExists.Result
            if (!bucketExists.Result)
            {
                log.LogInformation("Bucket: " + existingBucket + " does not exist or is inaccessible to the application");
                return false;
            }
            // Set up the Transfer Utility
            TransferUtility fileTransferUtility = new TransferUtility(myClient);

            // Stream the file
            try
            {

                log.LogInformation("Starting Copy");

                using (var stream = await blob.OpenReadAsync())
                {
                    // Note: You need permissions on the source blob
                    log.LogInformation("Streaming start...");
                   await  fileTransferUtility.UploadAsync(stream, existingBucket, keyName);
                    long fsize = stream.Length;
                    log.LogInformation("Streaming Done. Bytes=" + fsize.ToString());
                }

                log.LogInformation("Copy completed");
                // cannot get metadata as we have no rights on the S3: to compare filesize uploaded
                // var meta= fileTransferUtility.S3Client.GetObjectMetadataAsync(existingBucket, keyName);
                //  Int64 pgpfilesize = meta.Result.ContentLength; //compare to input file
                //  GetObjectMetadataRequest metadataRequest = new GetObjectMetadataRequest(bucketName, fileName);
                //  GetObjectMetadataResponse objectMetadata = meta.Result;
                // long contentLength = objectMetadata.ContentLength;

            }
            catch (AmazonS3Exception e)
            {
                log.LogInformation("Error encountered on server. Message:'{0}' when writing an object", e.Message);
            }
            catch (Exception e)
            {
                log.LogInformation("Unknown encountered on server. Message:'{0}' when writing an object", e.Message);
                return false;
            }

            return true;
        }

        public class ADFSET
        {
            public string SettingName { get; set; }
            public string SettingVal { get; set; }
        }

        public class ADFRequest
        {
            public List<ADFSET> ADFsettings { get; set; }
        }

        class ADFparams
        {
            public List<string> ParamNames { get; set; }
        }

        public class Result
        {
            public string result;
            public bool outcome;
            public string UTCtime;
            public string details;

            public Result(string msg, bool outcomeBool, string fullMsg)
            {
                result = msg;
                UTCtime = DateTime.Now.ToString("yyyy-MM-dd h:mm:ss tt");
                outcome = outcomeBool;
                details = fullMsg;
            }

            public string ConvertResultToJson()
            {
                return JsonConvert.SerializeObject(this);
            }
        }

        // trigger ---------------------------------------------------------------------------------
        [FunctionName("TransferAWS_HttpStart")]
        public static async Task<HttpResponseMessage> HttpStart(
            [HttpTrigger(AuthorizationLevel.Anonymous, "post")] HttpRequestMessage req,
            [DurableClient] IDurableOrchestrationClient starter,
            ILogger log)
        {
            req.Content.Headers.Remove("Content-Type");
            req.Content.Headers.Add("Content-Type", "application/json");
            var x = req.Content.ReadAsStringAsync();
            var data = await req.Content.ReadAsAsync<ADFRequest>();

            // Function input comes from the request content.
            string instanceId = await starter.StartNewAsync("TransferAWS", data);

            log.LogInformation($"Started orchestration with ID = '{instanceId}'.");

            return starter.CreateCheckStatusResponse(req, instanceId);
        }
    }
}